from pathlib import Path
from typing import Optional, Union
from nicegui import ui, app, App
from watchfiles import awatch
import hashlib
from starlette.middleware.base import BaseHTTPMiddleware

def _generate_data_id(url_path: str) -> str:
    md5_hash = hashlib.md5(url_path.encode("utf-8")).hexdigest()
    return md5_hash


def add_live_css(
    *,
    local_file: Union[str, Path],
    url_path: Optional[str] = None,
):
    """Add a live-reloading CSS file to the page.

    This method adds a CSS file to the page and updates the interface style without requiring a server restart whenever the file content changes.

    Args:
        local_file: The path to the local file.
        url_path: The URL path to serve the file from. If not provided, the file name will be used.

    """
    url_path = app.add_static_file(local_file=local_file, url_path=url_path)
    data_id = _generate_data_id(url_path)
    ui.add_head_html(f'<link data-id="{data_id}" rel="stylesheet" href="{url_path}"')

    ui.add_body_html(
        """
        <script>
            function reloadCSS(linkId) {
                const linkElement = document.querySelector(`link[data-id="${linkId}"]`);
                if (linkElement) {
                    const newHref = linkElement.href.split('?')[0] + '?reload=' + new Date().getTime();
                    const newLinkElement = document.createElement('link');
                    newLinkElement.rel = 'stylesheet';
                    newLinkElement.href = newHref;
                    newLinkElement.dataset.id = linkId;

                    // Replace the old link with the new one
                    linkElement.parentNode.insertBefore(newLinkElement, linkElement.nextSibling);
                    linkElement.parentNode.removeChild(linkElement);
                    console.log('Reloaded CSS:', linkElement.href);
                } else {
                    console.error('Link element with the specified ID not found.');
                }
            }
        </script>
    """
    )

    async def watch_file():
        async for changes in awatch(local_file):
            ui.run_javascript(f"reloadCSS('{data_id}')")
            continue

    @ui.context.client.on_connect
    async def on_connect():
        await watch_file()


def apply_nocache_css(app: App):
    """
    This function applies a middleware to the app that sets the Cache-Control and Pragma headers to prevent caching of CSS files.

    """

    class NoCacheCSSMiddleware(BaseHTTPMiddleware):
        async def dispatch(self, request, call_next):
            response = await call_next(request)
            if request["type"] == "http" and request["path"].endswith(".css"):
                response.headers["Cache-Control"] = (
                    "no-store, no-cache, must-revalidate, max-age=0"
                )
                response.headers["Pragma"] = "no-cache"
            return response

    app.add_middleware(NoCacheCSSMiddleware)


def detect_overflow_scroll(shared=False):
    """
    This function adds a CSS rule to detect if the element can scroll.
    """
    ui.add_head_html(
        """
<style>
body * {
    --can-scroll: 0;
    animation: detect-scroll;
    animation-timeline: scroll(self);
}

* {
    outline: calc(var(--can-scroll) * 10px) double rgb(232, 124, 48);
}


@keyframes detect-scroll {

    from,
    to {
        --can-scroll: 1;
    }
}
</style>
""",
        shared=shared,
    )
